<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\API\Dashboard\AuthController;
use App\Http\Controllers\API\Dashboard\UserController;
use App\Http\Controllers\API\Dashboard\RoleController;
use App\Http\Controllers\API\Dashboard\PermissionController;
use App\Http\Controllers\API\Dashboard\CityController;
use App\Http\Controllers\API\Dashboard\ZoneController;
use App\Http\Controllers\API\Dashboard\SettingController;
use App\Http\Controllers\API\Dashboard\GeneralDataController;
use App\Http\Controllers\API\Dashboard\FAQController;
use App\Http\Controllers\API\Dashboard\DashboardController;
use App\Http\Controllers\API\Dashboard\AnalyticsController;
use App\Http\Controllers\API\Client\AuthController as ClientAuthController;

// Client API Routes (Public)
Route::prefix('v1')->group(function () {
    Route::prefix('client')->group(function () {
        Route::post('/register-initial', [ClientAuthController::class, 'registerInitial']);
        Route::post('/verify-otp', [ClientAuthController::class, 'verifyOTP']);
        Route::post('/resend-otp', [ClientAuthController::class, 'resendOTP']);
        Route::post('/complete-profile', [ClientAuthController::class, 'completeProfile']);
        Route::post('/login', [ClientAuthController::class, 'login']);
        Route::post('/forgot-password', [ClientAuthController::class, 'forgotPassword']);
        Route::post('/reset-password', [ClientAuthController::class, 'resetPassword']);
        
        // Public routes for cities and zones
        Route::get('/cities', [CityController::class, 'index']);
        Route::get('/zones/city/{cityId}', [ZoneController::class, 'getByCity']);
        
        // Protected routes (require authentication)
        Route::middleware('auth:sanctum')->group(function () {
            Route::get('/profile', [ClientAuthController::class, 'getProfile']);
            Route::put('/profile', [ClientAuthController::class, 'updateProfile']);
        });
    });
});

// Dashboard API Routes
Route::prefix('v1')->group(function () {
    Route::prefix('dashboard')->group(function () {
        // Authentication Routes (Public)
    Route::post('/login', [AuthController::class, 'login']);
    Route::post('/forgot-password', [AuthController::class, 'forgotPassword']);
    Route::post('/reset-password', [AuthController::class, 'resetPassword']);
    
    // Protected Routes (require authentication)
    Route::middleware('auth:sanctum')->group(function () {
     Route::get('/user', function (Request $request) {
            $user = $request->user();
            $roles = $user->getRoleNames();
            $permissions = $user->getAllPermissions()->pluck('name');
            
            return response()->json([
                'success' => true,
                'data' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'type' => $user->type,
                    'status' => $user->status,
                    'roles' => $roles,
                    'permissions' => $permissions,
                ],
            ]);
        });
        Route::post('/logout', [AuthController::class, 'logout']);
        Route::post('/change-password', [AuthController::class, 'changePassword']);
        
        // Dashboard Statistics Route
        Route::get('/statistics', [DashboardController::class, 'getStatistics']);
        
        // Analytics Route
        Route::get('/analytics', [AnalyticsController::class, 'getAnalytics']);
        
        // User Management Routes
        Route::prefix('users')->group(function () {
            Route::get('/', [UserController::class, 'index']);
            Route::get('/roles', [UserController::class, 'getRoles']);
            Route::post('/', [UserController::class, 'store']);
            Route::get('/{id}', [UserController::class, 'show']);
            Route::put('/{id}', [UserController::class, 'update']);
            Route::delete('/{id}', [UserController::class, 'destroy']);
            Route::patch('/{id}/status', [UserController::class, 'changeStatus']);
        });
        
        // Roles Management Routes
        Route::prefix('roles')->group(function () {
            Route::get('/', [RoleController::class, 'index']);
            Route::get('/permissions', [RoleController::class, 'getAllPermissions']);
            Route::post('/', [RoleController::class, 'store']);
            Route::get('/{id}', [RoleController::class, 'show']);
            Route::put('/{id}', [RoleController::class, 'update']);
            Route::delete('/{id}', [RoleController::class, 'destroy']);
            Route::post('/{id}/permissions/assign', [RoleController::class, 'assignPermissions']);
            Route::post('/{id}/permissions/revoke', [RoleController::class, 'revokePermissions']);
        });
        
        // Permissions Management Routes
        Route::prefix('permissions')->group(function () {
            Route::get('/', [PermissionController::class, 'index']);
            Route::post('/', [PermissionController::class, 'store']);
            Route::post('/bulk', [PermissionController::class, 'bulkCreate']);
            Route::get('/{id}', [PermissionController::class, 'show']);
            Route::put('/{id}', [PermissionController::class, 'update']);
            Route::delete('/{id}', [PermissionController::class, 'destroy']);
        });
        
        // Cities Management Routes
        Route::prefix('cities')->group(function () {
            Route::get('/', [CityController::class, 'index']);
            Route::post('/', [CityController::class, 'store']);
            Route::get('/{id}', [CityController::class, 'show']);
            Route::put('/{id}', [CityController::class, 'update']);
            Route::delete('/{id}', [CityController::class, 'destroy']);
        });
        
        // Zones Management Routes
        Route::prefix('zones')->group(function () {
            Route::get('/', [ZoneController::class, 'index']);
            Route::post('/', [ZoneController::class, 'store']);
            Route::get('/{id}', [ZoneController::class, 'show']);
            Route::put('/{id}', [ZoneController::class, 'update']);
            Route::delete('/{id}', [ZoneController::class, 'destroy']);
            Route::get('/city/{cityId}', [ZoneController::class, 'getByCity']);
        });
        
        // Settings Management Routes
        Route::prefix('settings')->group(function () {
            Route::get('/app', [SettingController::class, 'getAppSettings']);
            Route::get('/seo', [SettingController::class, 'getSeoSettings']);
            Route::put('/app', [SettingController::class, 'updateAppSettings']);
            Route::put('/seo', [SettingController::class, 'updateSeoSettings']);
        });
        
        // General Data Management Routes
        Route::prefix('general-data')->group(function () {
            Route::get('/contact', [GeneralDataController::class, 'getContactInfo']);
            Route::put('/contact', [GeneralDataController::class, 'updateContactInfo']);
            Route::get('/services', [GeneralDataController::class, 'getServices']);
            Route::put('/services', [GeneralDataController::class, 'updateServices']);
            Route::get('/features', [GeneralDataController::class, 'getFeatures']);
            Route::put('/features', [GeneralDataController::class, 'updateFeatures']);
            Route::get('/working-steps', [GeneralDataController::class, 'getWorkingSteps']);
            Route::put('/working-steps', [GeneralDataController::class, 'updateWorkingSteps']);
            Route::get('/about-us', [GeneralDataController::class, 'getAboutUs']);
            Route::put('/about-us', [GeneralDataController::class, 'updateAboutUs']);
            Route::get('/terms-and-conditions', [GeneralDataController::class, 'getTermsAndConditions']);
            Route::put('/terms-and-conditions', [GeneralDataController::class, 'updateTermsAndConditions']);
            Route::get('/privacy-policy', [GeneralDataController::class, 'getPrivacyPolicy']);
            Route::put('/privacy-policy', [GeneralDataController::class, 'updatePrivacyPolicy']);
            Route::get('/statistics', [GeneralDataController::class, 'getStatistics']);
            Route::put('/statistics', [GeneralDataController::class, 'updateStatistics']);
        });
        
        // FAQ Management Routes
        Route::prefix('faqs')->group(function () {
            Route::get('/', [FAQController::class, 'index']);
            Route::post('/', [FAQController::class, 'store']);
            Route::get('/{id}', [FAQController::class, 'show']);
            Route::put('/{id}', [FAQController::class, 'update']);
            Route::delete('/{id}', [FAQController::class, 'destroy']);
        });
    });
});
});
