<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Mail\UserCredentialsMail;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    use ApiResponse;

    /**
     * عرض قائمة المستخدمين
     */
    public function index(Request $request)
    {
        $query = User::with(['roles', 'city', 'zone']);

        // البحث
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        // التصفية حسب النوع
        if ($request->has('type')) {
            $query->where('type', $request->type);
        }

        // التصفية حسب الحالة
        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $users = $query->paginate($perPage);

        return $this->successResponse($users, __('messages.users_listed'));
    }

    /**
     * إنشاء مستخدم جديد
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'middle_name' => 'required|string|max:255',
                'third_name' => 'nullable|string|max:255',
                'last_name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'phone' => 'required|string|unique:users,phone',
                'whatsapp' => 'required|string|unique:users,whatsapp',
                'password' => 'required|string|min:8',
                 'city_id' => 'nullable|exists:cities,id',
                'zone_id' => 'nullable|exists:zones,id',
                 'role_id' => 'nullable|exists:roles,id',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // إنشاء كلمة مرور عشوائية إذا لم يتم إدخالها
        $password = $request->password ?? Str::random(12);

        // إنشاء المستخدم
        $user = User::create([
            'name' => $request->name,
            'middle_name' => $request->middle_name,
            'third_name' => $request->third_name ?? '',
            'last_name' => $request->last_name,
            'email' => $request->email,
            'phone' => $request->phone,
            'whatsapp' => $request->whatsapp,
            'password' => Hash::make($password),
              'address' => $request->address,
            'city_id' => $request->city_id,
            'zone_id' => $request->zone_id,
             'type' => 'user',
            'status' => $request->status ?? 'active',
            'is_verified' => true,
        ]);

        // إعطاء الدور إذا تم تحديده
        if ($request->has('role_id')) {
            $role = Role::find($request->role_id);
            if ($role) {
                $user->assignRole($role);
            }
        }

        // إرسال الإيميل ببيانات المستخدم
        try {
            $loginUrl = config('app.frontend_url', config('app.url')) . '/login';
            Mail::to($user->email)->send(new UserCredentialsMail(
                $user->email,
                $password,
                $loginUrl,
                $user->name
            ));
        } catch (\Exception $e) {
            // لا نوقف العملية إذا فشل الإرسال
            \Log::error('Failed to send user credentials email: ' . $e->getMessage());
        }

        return $this->successResponse([
            'user' => $user->load('roles'),
        ], __('messages.user_created'));
    }

    /**
     * عرض مستخدم محدد
     */
    public function show($id)
    {
        $user = User::with(['roles', 'city', 'zone'])->find($id);

        if (!$user) {
            return $this->notFoundResponse(__('messages.user_not_found'));
        }

        return $this->successResponse($user, __('messages.user_retrieved'));
    }

    /**
     * تحديث مستخدم
     */
    public function update(Request $request, $id)
    {
        $user = User::find($id);

        if (!$user) {
            return $this->notFoundResponse(__('messages.user_not_found'));
        }

        try {
            $validated = $request->validate([
                'name' => 'sometimes|required|string|max:255',
                'middle_name' => 'sometimes|required|string|max:255',
                'third_name' => 'nullable|string|max:255',
                'last_name' => 'sometimes|required|string|max:255',
                'email' => 'sometimes|required|email|unique:users,email,' . $id,
                'phone' => 'sometimes|required|string|unique:users,phone,' . $id,
                'whatsapp' => 'sometimes|required|string|unique:users,whatsapp,' . $id,
                  'city_id' => 'nullable|exists:cities,id',
                'zone_id' => 'nullable|exists:zones,id',
                 'status' => 'nullable|in:active,inactive',
                'role_id' => 'nullable|exists:roles,id',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // تحديث البيانات
        $updateData = $request->only([
            'name', 'middle_name', 'third_name', 'last_name',
            'email', 'phone', 'whatsapp',   'address', 'city_id', 'zone_id',
            'status'
        ]);

        

        $user->update($updateData);

        // تحديث الدور
        if ($request->has('role_id')) {
            $user->roles()->detach();
            $role = Role::find($request->role_id);
            if ($role) {
                $user->assignRole($role);
            }
        }

        return $this->successResponse([
            'user' => $user->load('roles', 'city', 'zone'),
        ], __('messages.user_updated'));
    }

    /**
     * حذف مستخدم
     */
    public function destroy($id)
    {
        $user = User::find($id);

        if (!$user) {
            return $this->notFoundResponse(__('messages.user_not_found'));
        }

        // منع حذف المستخدم الحالي
        if ($user->id === auth()->id()) {
            return $this->errorResponse(__('messages.cannot_delete_self'), 400);
        }

        $user->delete();

        return $this->successResponse(null, __('messages.user_deleted'));
    }

    /**
     * تغيير حالة الحساب
     */
    public function changeStatus(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'status' => 'required|in:active,inactive',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $user = User::find($id);

        if (!$user) {
            return $this->notFoundResponse(__('messages.user_not_found'));
        }

        $user->update([
            'status' => $request->status,
        ]);

        return $this->successResponse([
            'user' => $user,
        ], __('messages.user_status_updated'));
    }

    /**
     * جلب قائمة الأدوار
     */
    public function getRoles()
    {
        $roles = Role::all();
        return $this->successResponse($roles, __('messages.roles_retrieved'));
    }
}
