<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\File;

class SettingController extends Controller
{
    use ApiResponse;

    /**
     * الحصول على مسار مجلد uploads
     */
    private function getUploadsPath()
    {
        $uploadsPath = public_path('uploads');
        
        // إنشاء المجلد إذا لم يكن موجوداً
        if (!File::exists($uploadsPath)) {
            File::makeDirectory($uploadsPath, 0755, true);
        }
        
        return $uploadsPath;
    }

    /**
     * الحصول على URL للصورة
     */
    private function getImageUrl($filename)
    {
        if (empty($filename)) {
            return '';
        }
        
        // إذا كان المسار يحتوي على uploads، نعيد URL مباشر
        if (strpos($filename, 'uploads/') !== false) {
            return url($filename);
        }
        
        // إذا كان اسم ملف فقط، نضيف uploads/
        return url('uploads/' . $filename);
    }

    /**
     * عرض بيانات التطبيق
     */
    public function getAppSettings()
    {
        $settings = Setting::first();

        if (!$settings) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $settings = Setting::create([
                'logo' => '',
                'icon' => '',
                'favicon' => '',
                'app_name_en' => 'Application',
                'app_name_ar' => 'التطبيق',
                'app_description_en' => '',
                'app_description_ar' => '',
            ]);
        }

        // إرجاع URLs كاملة للصور
        $appSettings = [
            'logo' => $this->getImageUrl($settings->logo),
            'icon' => $this->getImageUrl($settings->icon),
            'favicon' => $this->getImageUrl($settings->favicon),
            'app_name_en' => $settings->app_name_en,
            'app_name_ar' => $settings->app_name_ar,
            'app_description_en' => $settings->app_description_en,
            'app_description_ar' => $settings->app_description_ar,
        ];

        return $this->successResponse($appSettings, __('messages.app_settings_retrieved'));
    }

    /**
     * عرض بيانات السيو
     */
    public function getSeoSettings()
    {
        $settings = Setting::first();

        if (!$settings) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $settings = Setting::create([
                'logo' => '',
                'icon' => '',
                'favicon' => '',
                'app_name_en' => 'Application',
                'app_name_ar' => 'التطبيق',
            ]);
        }

        $seoSettings = [
            'seo_title_en' => $settings->seo_title_en,
            'seo_title_ar' => $settings->seo_title_ar,
            'seo_description_en' => $settings->seo_description_en,
            'seo_description_ar' => $settings->seo_description_ar,
            'seo_keywords_en' => $settings->seo_keywords_en,
            'seo_keywords_ar' => $settings->seo_keywords_ar,
        ];

        return $this->successResponse($seoSettings, __('messages.seo_settings_retrieved'));
    }

    /**
     * تحديث بيانات التطبيق
     */
    public function updateAppSettings(Request $request)
    {
        try {
            $validated = $request->validate([
                'logo' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:5120',
                'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,ico|max:2048',
                'favicon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg,ico|max:1024',
                'app_name_en' => 'sometimes|required|string|max:255',
                'app_name_ar' => 'sometimes|required|string|max:255',
                'app_description_en' => 'nullable|string',
                'app_description_ar' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $settings = Setting::first();

        if (!$settings) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $settings = Setting::create([
                'logo' => '',
                'icon' => '',
                'favicon' => '',
                'app_name_en' => $request->app_name_en ?? 'Application',
                'app_name_ar' => $request->app_name_ar ?? 'التطبيق',
                'app_description_en' => $request->app_description_en ?? '',
                'app_description_ar' => $request->app_description_ar ?? '',
            ]);
        }

        // تحديث البيانات الموجودة
        $updateData = [];
        $uploadsPath = $this->getUploadsPath();

        // معالجة رفع اللوجو
        if ($request->hasFile('logo')) {
            // حذف الصورة القديمة إن وجدت
            if ($settings->logo) {
                $oldLogoPath = $settings->logo;
                // إذا كان المسار يحتوي على uploads/، نستخدمه مباشرة
                if (strpos($oldLogoPath, 'uploads/') === 0) {
                    $oldLogoPath = public_path($oldLogoPath);
                } else {
                    $oldLogoPath = public_path('uploads/' . $oldLogoPath);
                }
                
                if (File::exists($oldLogoPath)) {
                    File::delete($oldLogoPath);
                }
            }
            
            $logoFile = $request->file('logo');
            $logoName = 'logo_' . time() . '.' . $logoFile->getClientOriginalExtension();
            $logoFile->move($uploadsPath, $logoName);
            $updateData['logo'] = 'uploads/' . $logoName;
        } elseif ($request->has('logo') && is_string($request->logo)) {
            // إذا كان نص (URL موجود)
            $updateData['logo'] = $request->logo;
        }

        // معالجة رفع الأيقونة
        if ($request->hasFile('icon')) {
            // حذف الصورة القديمة إن وجدت
            if ($settings->icon) {
                $oldIconPath = $settings->icon;
                // إذا كان المسار يحتوي على uploads/، نستخدمه مباشرة
                if (strpos($oldIconPath, 'uploads/') === 0) {
                    $oldIconPath = public_path($oldIconPath);
                } else {
                    $oldIconPath = public_path('uploads/' . $oldIconPath);
                }
                
                if (File::exists($oldIconPath)) {
                    File::delete($oldIconPath);
                }
            }
            
            $iconFile = $request->file('icon');
            $iconName = 'icon_' . time() . '.' . $iconFile->getClientOriginalExtension();
            $iconFile->move($uploadsPath, $iconName);
            $updateData['icon'] = 'uploads/' . $iconName;
        } elseif ($request->has('icon') && is_string($request->icon)) {
            // إذا كان نص (URL موجود)
            $updateData['icon'] = $request->icon;
        }

        // معالجة رفع Favicon
        if ($request->hasFile('favicon')) {
            // حذف الصورة القديمة إن وجدت
            if ($settings->favicon) {
                $oldFaviconPath = $settings->favicon;
                // إذا كان المسار يحتوي على uploads/، نستخدمه مباشرة
                if (strpos($oldFaviconPath, 'uploads/') === 0) {
                    $oldFaviconPath = public_path($oldFaviconPath);
                } else {
                    $oldFaviconPath = public_path('uploads/' . $oldFaviconPath);
                }
                
                if (File::exists($oldFaviconPath)) {
                    File::delete($oldFaviconPath);
                }
            }
            
            $faviconFile = $request->file('favicon');
            $faviconName = 'favicon_' . time() . '.' . $faviconFile->getClientOriginalExtension();
            $faviconFile->move($uploadsPath, $faviconName);
            $updateData['favicon'] = 'uploads/' . $faviconName;
        } elseif ($request->has('favicon') && is_string($request->favicon)) {
            // إذا كان نص (URL موجود)
            $updateData['favicon'] = $request->favicon;
        }

        // تحديث النصوص
        if ($request->has('app_name_en')) {
            $updateData['app_name_en'] = $request->app_name_en;
        }
        if ($request->has('app_name_ar')) {
            $updateData['app_name_ar'] = $request->app_name_ar;
        }
        if ($request->has('app_description_en')) {
            $updateData['app_description_en'] = $request->app_description_en;
        }
        if ($request->has('app_description_ar')) {
            $updateData['app_description_ar'] = $request->app_description_ar;
        }

        $settings->update($updateData);

        // إرجاع URLs كاملة للصور
        $appSettings = [
            'logo' => $this->getImageUrl($settings->logo),
            'icon' => $this->getImageUrl($settings->icon),
            'favicon' => $this->getImageUrl($settings->favicon),
            'app_name_en' => $settings->app_name_en,
            'app_name_ar' => $settings->app_name_ar,
            'app_description_en' => $settings->app_description_en,
            'app_description_ar' => $settings->app_description_ar,
        ];

        return $this->successResponse($appSettings, __('messages.app_settings_updated'));
    }

    /**
     * تحديث بيانات السيو
     */
    public function updateSeoSettings(Request $request)
    {
        try {
            $validated = $request->validate([
                'seo_title_en' => 'nullable|string|max:255',
                'seo_title_ar' => 'nullable|string|max:255',
                'seo_description_en' => 'nullable|string',
                'seo_description_ar' => 'nullable|string',
                'seo_keywords_en' => 'nullable|string',
                'seo_keywords_ar' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $settings = Setting::first();

        if (!$settings) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $settings = Setting::create([
                'logo' => '',
                'icon' => '',
                'favicon' => '',
                'app_name_en' => 'Application',
                'app_name_ar' => 'التطبيق',
                'seo_title_en' => $request->seo_title_en,
                'seo_title_ar' => $request->seo_title_ar,
                'seo_description_en' => $request->seo_description_en,
                'seo_description_ar' => $request->seo_description_ar,
                'seo_keywords_en' => $request->seo_keywords_en,
                'seo_keywords_ar' => $request->seo_keywords_ar,
            ]);
        } else {
            // تحديث البيانات الموجودة
            $updateData = [];
            
            if ($request->has('seo_title_en')) {
                $updateData['seo_title_en'] = $request->seo_title_en;
            }
            if ($request->has('seo_title_ar')) {
                $updateData['seo_title_ar'] = $request->seo_title_ar;
            }
            if ($request->has('seo_description_en')) {
                $updateData['seo_description_en'] = $request->seo_description_en;
            }
            if ($request->has('seo_description_ar')) {
                $updateData['seo_description_ar'] = $request->seo_description_ar;
            }
            if ($request->has('seo_keywords_en')) {
                $updateData['seo_keywords_en'] = $request->seo_keywords_en;
            }
            if ($request->has('seo_keywords_ar')) {
                $updateData['seo_keywords_ar'] = $request->seo_keywords_ar;
            }

            $settings->update($updateData);
        }

        $seoSettings = [
            'seo_title_en' => $settings->seo_title_en,
            'seo_title_ar' => $settings->seo_title_ar,
            'seo_description_en' => $settings->seo_description_en,
            'seo_description_ar' => $settings->seo_description_ar,
            'seo_keywords_en' => $settings->seo_keywords_en,
            'seo_keywords_ar' => $settings->seo_keywords_ar,
        ];

        return $this->successResponse($seoSettings, __('messages.seo_settings_updated'));
    }
}
