<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RoleController extends Controller
{
    use ApiResponse;

    /**
     * عرض قائمة الأدوار
     */
    public function index(Request $request)
    {
        $query = Role::with('permissions');

        // البحث
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $roles = $query->paginate($perPage);

        return $this->successResponse($roles, __('messages.roles_listed'));
    }

    /**
     * إنشاء دور جديد
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255|unique:roles,name',
                'guard_name' => 'nullable|string|in:web,api',
                'permissions' => 'nullable|array',
                'permissions.*' => 'exists:permissions,id',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // إنشاء الدور
        $role = Role::create([
            'name' => $request->name,
            'guard_name' => $request->guard_name ?? 'api',
        ]);

        // إعطاء الصلاحيات إذا تم تحديدها
        if ($request->has('permissions') && is_array($request->permissions)) {
            $permissions = Permission::whereIn('id', $request->permissions)->get();
            $role->syncPermissions($permissions);
        }

        return $this->successResponse([
            'role' => $role->load('permissions'),
        ], __('messages.role_created'));
    }

    /**
     * عرض دور محدد
     */
    public function show($id)
    {
        $role = Role::with('permissions')->find($id);

        if (!$role) {
            return $this->notFoundResponse(__('messages.role_not_found'));
        }

        return $this->successResponse($role, __('messages.role_retrieved'));
    }

    /**
     * تحديث دور
     */
    public function update(Request $request, $id)
    {
        $role = Role::find($id);

        if (!$role) {
            return $this->notFoundResponse(__('messages.role_not_found'));
        }

        try {
            $validated = $request->validate([
                'name' => 'sometimes|required|string|max:255|unique:roles,name,' . $id,
                'guard_name' => 'nullable|string|in:web,api',
                'permissions' => 'nullable|array',
                'permissions.*' => 'exists:permissions,id',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // تحديث البيانات
        $updateData = [];
        if ($request->has('name')) {
            $updateData['name'] = $request->name;
        }
        if ($request->has('guard_name')) {
            $updateData['guard_name'] = $request->guard_name;
        }

        if (!empty($updateData)) {
            $role->update($updateData);
        }

        // تحديث الصلاحيات إذا تم تحديدها
        if ($request->has('permissions')) {
            if (is_array($request->permissions) && count($request->permissions) > 0) {
                $permissions = Permission::whereIn('id', $request->permissions)->get();
                $role->syncPermissions($permissions);
            } else {
                // إذا كان المصفوفة فارغة، نزيل كل الصلاحيات
                $role->syncPermissions([]);
            }
        }

        return $this->successResponse([
            'role' => $role->load('permissions'),
        ], __('messages.role_updated'));
    }

    /**
     * حذف دور
     */
    public function destroy($id)
    {
        $role = Role::find($id);

        if (!$role) {
            return $this->notFoundResponse(__('messages.role_not_found'));
        }

        // منع حذف الأدوار الأساسية إذا لزم الأمر
        // يمكنك إضافة شروط أخرى هنا

        $role->delete();

        return $this->successResponse(null, __('messages.role_deleted'));
    }

    /**
     * إعطاء صلاحيات لدور
     */
    public function assignPermissions(Request $request, $id)
    {
        $role = Role::find($id);

        if (!$role) {
            return $this->notFoundResponse(__('messages.role_not_found'));
        }

        try {
            $validated = $request->validate([
                'permissions' => 'required|array',
                'permissions.*' => 'exists:permissions,id',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $permissions = Permission::whereIn('id', $request->permissions)->get();
        $role->syncPermissions($permissions);

        return $this->successResponse([
            'role' => $role->load('permissions'),
        ], __('messages.permissions_assigned_to_role'));
    }

    /**
     * إزالة صلاحيات من دور
     */
    public function revokePermissions(Request $request, $id)
    {
        $role = Role::find($id);

        if (!$role) {
            return $this->notFoundResponse(__('messages.role_not_found'));
        }

        try {
            $validated = $request->validate([
                'permissions' => 'required|array',
                'permissions.*' => 'exists:permissions,id',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $permissions = Permission::whereIn('id', $request->permissions)->get();
        $role->revokePermissionTo($permissions);

        return $this->successResponse([
            'role' => $role->load('permissions'),
        ], __('messages.permissions_revoked_from_role'));
    }

    /**
     * جلب جميع الصلاحيات (للاستخدام في النماذج)
     */
    public function getAllPermissions()
    {
        $permissions = Permission::orderBy('name')->get();
        return $this->successResponse($permissions, __('messages.permissions_retrieved'));
    }
}
