<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Spatie\Permission\Models\Permission;

class PermissionController extends Controller
{
    use ApiResponse;

    /**
     * عرض قائمة الصلاحيات
     */
    public function index(Request $request)
    {
        $query = Permission::query();

        // البحث
        if ($request->has('search')) {
            $search = $request->search;
            $query->where('name', 'like', "%{$search}%");
        }

        // التصفية حسب guard_name
        if ($request->has('guard_name')) {
            $query->where('guard_name', $request->guard_name);
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'name');
        $sortOrder = $request->get('sort_order', 'asc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $permissions = $query->paginate($perPage);

        return $this->successResponse($permissions, __('messages.permissions_listed'));
    }

    /**
     * إنشاء صلاحية جديدة
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255|unique:permissions,name',
                'guard_name' => 'nullable|string|in:web,api',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // إنشاء الصلاحية
        $permission = Permission::create([
            'name' => $request->name,
            'guard_name' => $request->guard_name ?? 'api',
        ]);

        return $this->successResponse([
            'permission' => $permission,
        ], __('messages.permission_created'));
    }

    /**
     * عرض صلاحية محددة
     */
    public function show($id)
    {
        $permission = Permission::find($id);

        if (!$permission) {
            return $this->notFoundResponse(__('messages.permission_not_found'));
        }

        return $this->successResponse($permission, __('messages.permission_retrieved'));
    }

    /**
     * تحديث صلاحية
     */
    public function update(Request $request, $id)
    {
        $permission = Permission::find($id);

        if (!$permission) {
            return $this->notFoundResponse(__('messages.permission_not_found'));
        }

        try {
            $validated = $request->validate([
                'name' => 'sometimes|required|string|max:255|unique:permissions,name,' . $id,
                'guard_name' => 'nullable|string|in:web,api',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // تحديث البيانات
        $updateData = [];
        if ($request->has('name')) {
            $updateData['name'] = $request->name;
        }
        if ($request->has('guard_name')) {
            $updateData['guard_name'] = $request->guard_name;
        }

        if (!empty($updateData)) {
            $permission->update($updateData);
        }

        return $this->successResponse([
            'permission' => $permission,
        ], __('messages.permission_updated'));
    }

    /**
     * حذف صلاحية
     */
    public function destroy($id)
    {
        $permission = Permission::find($id);

        if (!$permission) {
            return $this->notFoundResponse(__('messages.permission_not_found'));
        }

        $permission->delete();

        return $this->successResponse(null, __('messages.permission_deleted'));
    }

    /**
     * إنشاء عدة صلاحيات دفعة واحدة
     */
    public function bulkCreate(Request $request)
    {
        try {
            $validated = $request->validate([
                'permissions' => 'required|array',
                'permissions.*.name' => 'required|string|max:255',
                'permissions.*.guard_name' => 'nullable|string|in:web,api',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $createdPermissions = [];
        foreach ($request->permissions as $permissionData) {
            $permission = Permission::firstOrCreate(
                [
                    'name' => $permissionData['name'],
                    'guard_name' => $permissionData['guard_name'] ?? 'api',
                ]
            );
            $createdPermissions[] = $permission;
        }

        return $this->successResponse([
            'permissions' => $createdPermissions,
            'count' => count($createdPermissions),
        ], __('messages.permissions_bulk_created'));
    }
}
