<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\ContactInfo;
use App\Models\GeneralInfo;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class GeneralDataController extends Controller
{
    use ApiResponse;

    /**
     * جلب بيانات التواصل
     */
    public function getContactInfo()
    {
        $contactInfo = ContactInfo::first();

        if (!$contactInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $contactInfo = ContactInfo::create([
                'phones' => [],
                'emails' => [],
                'addresses' => [],
                'working_hours' => [],
                'facebook' => null,
                'whatsapp' => null,
                'instegram' => null,
                'linkedin' => null,
                'youtube' => null,
                'threads' => null,
            ]);
        }

        // Laravel casts ستحول تلقائياً من JSON إلى مصفوفات
        $data = [
            'phones' => $contactInfo->phones ?? [],
            'emails' => $contactInfo->emails ?? [],
            'addresses' => $contactInfo->addresses ?? [],
            'working_hours' => $contactInfo->working_hours ?? [],
            'facebook' => $contactInfo->facebook,
            'whatsapp' => $contactInfo->whatsapp,
            'instegram' => $contactInfo->instegram,
            'linkedin' => $contactInfo->linkedin,
            'youtube' => $contactInfo->youtube,
            'threads' => $contactInfo->threads,
        ];

        return $this->successResponse($data, __('messages.contact_info_retrieved'));
    }

    /**
     * تحديث بيانات التواصل
     */
    public function updateContactInfo(Request $request)
    {
        try {
            $validated = $request->validate([
                'phones' => 'nullable|array',
                'phones.*' => 'string|max:255',
                'emails' => 'nullable|array',
                'emails.*' => 'email|max:255',
                'addresses' => 'nullable|array',
                'addresses.*.address_ar' => 'required_with:addresses.*|string',
                'addresses.*.address_en' => 'required_with:addresses.*|string',
                'working_hours' => 'nullable|array',
                'working_hours.*.working_hours_ar' => 'required_with:working_hours.*|string',
                'working_hours.*.working_hours_en' => 'required_with:working_hours.*|string',
                'facebook' => 'nullable|string|max:255|url',
                'whatsapp' => 'nullable|string|max:255|url',
                'instegram' => 'nullable|string|max:255|url',
                'linkedin' => 'nullable|string|max:255|url',
                'youtube' => 'nullable|string|max:255|url',
                'threads' => 'nullable|string|max:255|url',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $contactInfo = ContactInfo::first();

        $updateData = [];

        // تحديث المصفوفات (Laravel casts ستحولها تلقائياً إلى JSON)
        if ($request->has('phones')) {
            $updateData['phones'] = $request->phones;
        }
        if ($request->has('emails')) {
            $updateData['emails'] = $request->emails;
        }
        if ($request->has('addresses')) {
            $updateData['addresses'] = $request->addresses;
        }
        if ($request->has('working_hours')) {
            $updateData['working_hours'] = $request->working_hours;
        }

        // تحديث روابط السوشيال ميديا
        if ($request->has('facebook')) {
            $updateData['facebook'] = $request->facebook;
        }
        if ($request->has('whatsapp')) {
            $updateData['whatsapp'] = $request->whatsapp;
        }
        if ($request->has('instegram')) {
            $updateData['instegram'] = $request->instegram;
        }
        if ($request->has('linkedin')) {
            $updateData['linkedin'] = $request->linkedin;
        }
        if ($request->has('youtube')) {
            $updateData['youtube'] = $request->youtube;
        }
        if ($request->has('threads')) {
            $updateData['threads'] = $request->threads;
        }

        if (!$contactInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $contactInfo = ContactInfo::create([
                'phones' => $updateData['phones'] ?? [],
                'emails' => $updateData['emails'] ?? [],
                'addresses' => $updateData['addresses'] ?? [],
                'working_hours' => $updateData['working_hours'] ?? [],
                'facebook' => $updateData['facebook'] ?? null,
                'whatsapp' => $updateData['whatsapp'] ?? null,
                'instegram' => $updateData['instegram'] ?? null,
                'linkedin' => $updateData['linkedin'] ?? null,
                'youtube' => $updateData['youtube'] ?? null,
                'threads' => $updateData['threads'] ?? null,
            ]);
        } else {
            // تحديث البيانات الموجودة
            $contactInfo->update($updateData);
            $contactInfo->refresh(); // إعادة تحميل البيانات بعد التحديث
        }

        // إرجاع البيانات المحدثة كمصفوفات (Laravel casts ستحولها تلقائياً)
        $data = [
            'phones' => $contactInfo->phones ?? [],
            'emails' => $contactInfo->emails ?? [],
            'addresses' => $contactInfo->addresses ?? [],
            'working_hours' => $contactInfo->working_hours ?? [],
            'facebook' => $contactInfo->facebook,
            'whatsapp' => $contactInfo->whatsapp,
            'instegram' => $contactInfo->instegram,
            'linkedin' => $contactInfo->linkedin,
            'youtube' => $contactInfo->youtube,
            'threads' => $contactInfo->threads,
        ];

        return $this->successResponse($data, __('messages.contact_info_updated'));
    }

    /**
     * عرض بيانات الخدمات
     */
    public function getServices()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'services_en' => [],
                'services_ar' => [],
            ]);
        }

        // دمج الخدمات من العربية والإنجليزية
        $servicesEn = $generalInfo->services_en ?? [];
        $servicesAr = $generalInfo->services_ar ?? [];
        
        // دمج الخدمات في مصفوفة واحدة
        $services = [];
        $maxLength = max(count($servicesEn), count($servicesAr));
        
        for ($i = 0; $i < $maxLength; $i++) {
            $services[] = [
                'name_ar' => $servicesAr[$i]['name'] ?? '',
                'name_en' => $servicesEn[$i]['name'] ?? '',
                'description_ar' => $servicesAr[$i]['description'] ?? '',
                'description_en' => $servicesEn[$i]['description'] ?? '',
            ];
        }

        return $this->successResponse([
            'services' => $services,
        ], __('messages.services_retrieved'));
    }

    /**
     * تحديث بيانات الخدمات
     */
    public function updateServices(Request $request)
    {
        try {
            $validated = $request->validate([
                'services' => 'required|array',
                'services.*.name_ar' => 'required|string|max:255',
                'services.*.name_en' => 'required|string|max:255',
                'services.*.description_ar' => 'nullable|string',
                'services.*.description_en' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        // فصل الخدمات إلى مصفوفتين (عربي وإنجليزي)
        $servicesAr = [];
        $servicesEn = [];

        foreach ($request->services as $service) {
            $servicesAr[] = [
                'name' => $service['name_ar'],
                'description' => $service['description_ar'] ?? '',
            ];
            $servicesEn[] = [
                'name' => $service['name_en'],
                'description' => $service['description_en'] ?? '',
            ];
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'services_ar' => $servicesAr,
                'services_en' => $servicesEn,
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update([
                'services_ar' => $servicesAr,
                'services_en' => $servicesEn,
            ]);
            $generalInfo->refresh();
        }

        // إرجاع الخدمات المحدثة
        $services = [];
        foreach ($request->services as $service) {
            $services[] = [
                'name_ar' => $service['name_ar'],
                'name_en' => $service['name_en'],
                'description_ar' => $service['description_ar'] ?? '',
                'description_en' => $service['description_en'] ?? '',
            ];
        }

        return $this->successResponse([
            'services' => $services,
        ], __('messages.services_updated'));
    }

    /**
     * عرض بيانات الميزات
     */
    public function getFeatures()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'features_en' => [],
                'features_ar' => [],
            ]);
        }

        // دمج الميزات من العربية والإنجليزية
        $featuresEn = $generalInfo->features_en ?? [];
        $featuresAr = $generalInfo->features_ar ?? [];
        
        // دمج الميزات في مصفوفة واحدة
        $features = [];
        $maxLength = max(count($featuresEn), count($featuresAr));
        
        for ($i = 0; $i < $maxLength; $i++) {
            $features[] = [
                'name_ar' => $featuresAr[$i]['name'] ?? '',
                'name_en' => $featuresEn[$i]['name'] ?? '',
                'description_ar' => $featuresAr[$i]['description'] ?? '',
                'description_en' => $featuresEn[$i]['description'] ?? '',
            ];
        }

        return $this->successResponse([
            'features' => $features,
        ], __('messages.features_retrieved'));
    }

    /**
     * تحديث بيانات الميزات
     */
    public function updateFeatures(Request $request)
    {
        try {
            $validated = $request->validate([
                'features' => 'required|array',
                'features.*.name_ar' => 'required|string|max:255',
                'features.*.name_en' => 'required|string|max:255',
                'features.*.description_ar' => 'nullable|string',
                'features.*.description_en' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        // فصل الميزات إلى مصفوفتين (عربي وإنجليزي)
        $featuresAr = [];
        $featuresEn = [];

        foreach ($request->features as $feature) {
            $featuresAr[] = [
                'name' => $feature['name_ar'],
                'description' => $feature['description_ar'] ?? '',
            ];
            $featuresEn[] = [
                'name' => $feature['name_en'],
                'description' => $feature['description_en'] ?? '',
            ];
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'features_ar' => $featuresAr,
                'features_en' => $featuresEn,
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update([
                'features_ar' => $featuresAr,
                'features_en' => $featuresEn,
            ]);
            $generalInfo->refresh();
        }

        // إرجاع الميزات المحدثة
        $features = [];
        foreach ($request->features as $feature) {
            $features[] = [
                'name_ar' => $feature['name_ar'],
                'name_en' => $feature['name_en'],
                'description_ar' => $feature['description_ar'] ?? '',
                'description_en' => $feature['description_en'] ?? '',
            ];
        }

        return $this->successResponse([
            'features' => $features,
        ], __('messages.features_updated'));
    }

    /**
     * عرض بيانات خطوات كيف نعمل
     */
    public function getWorkingSteps()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'working_steps_en' => [],
                'working_steps_ar' => [],
            ]);
        }

        // دمج الخطوات من العربية والإنجليزية
        $workingStepsEn = $generalInfo->working_steps_en ?? [];
        $workingStepsAr = $generalInfo->working_steps_ar ?? [];
        
        // دمج الخطوات في مصفوفة واحدة
        $workingSteps = [];
        $maxLength = max(count($workingStepsEn), count($workingStepsAr));
        
        for ($i = 0; $i < $maxLength; $i++) {
            $workingSteps[] = [
                'name_ar' => $workingStepsAr[$i]['name'] ?? '',
                'name_en' => $workingStepsEn[$i]['name'] ?? '',
                'description_ar' => $workingStepsAr[$i]['description'] ?? '',
                'description_en' => $workingStepsEn[$i]['description'] ?? '',
            ];
        }

        return $this->successResponse([
            'working_steps' => $workingSteps,
        ], __('messages.working_steps_retrieved'));
    }

    /**
     * تحديث بيانات خطوات كيف نعمل
     */
    public function updateWorkingSteps(Request $request)
    {
        try {
            $validated = $request->validate([
                'working_steps' => 'required|array',
                'working_steps.*.name_ar' => 'required|string|max:255',
                'working_steps.*.name_en' => 'required|string|max:255',
                'working_steps.*.description_ar' => 'nullable|string',
                'working_steps.*.description_en' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        // فصل الخطوات إلى مصفوفتين (عربي وإنجليزي)
        $workingStepsAr = [];
        $workingStepsEn = [];

        foreach ($request->working_steps as $step) {
            $workingStepsAr[] = [
                'name' => $step['name_ar'],
                'description' => $step['description_ar'] ?? '',
            ];
            $workingStepsEn[] = [
                'name' => $step['name_en'],
                'description' => $step['description_en'] ?? '',
            ];
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'working_steps_ar' => $workingStepsAr,
                'working_steps_en' => $workingStepsEn,
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update([
                'working_steps_ar' => $workingStepsAr,
                'working_steps_en' => $workingStepsEn,
            ]);
            $generalInfo->refresh();
        }

        // إرجاع الخطوات المحدثة
        $workingSteps = [];
        foreach ($request->working_steps as $step) {
            $workingSteps[] = [
                'name_ar' => $step['name_ar'],
                'name_en' => $step['name_en'],
                'description_ar' => $step['description_ar'] ?? '',
                'description_en' => $step['description_en'] ?? '',
            ];
        }

        return $this->successResponse([
            'working_steps' => $workingSteps,
        ], __('messages.working_steps_updated'));
    }

    /**
     * عرض بيانات من نحن
     */
    public function getAboutUs()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'about_us_en' => '',
                'about_us_ar' => '',
                'message_en' => '',
                'message_ar' => '',
                'vision_en' => '',
                'vision_ar' => '',
                'values_en' => [],
                'values_ar' => [],
            ]);
        }

        // دمج القيم من العربية والإنجليزية
        $valuesEn = $generalInfo->values_en ?? [];
        $valuesAr = $generalInfo->values_ar ?? [];
        
        // دمج القيم في مصفوفة واحدة
        $values = [];
        $maxLength = max(count($valuesEn), count($valuesAr));
        
        for ($i = 0; $i < $maxLength; $i++) {
            $values[] = [
                'title_ar' => $valuesAr[$i]['title'] ?? '',
                'title_en' => $valuesEn[$i]['title'] ?? '',
                'description_ar' => $valuesAr[$i]['description'] ?? '',
                'description_en' => $valuesEn[$i]['description'] ?? '',
            ];
        }

        $data = [
            'about_us_ar' => $generalInfo->about_us_ar ?? '',
            'about_us_en' => $generalInfo->about_us_en ?? '',
            'message_ar' => $generalInfo->message_ar ?? '',
            'message_en' => $generalInfo->message_en ?? '',
            'vision_ar' => $generalInfo->vision_ar ?? '',
            'vision_en' => $generalInfo->vision_en ?? '',
            'values' => $values,
        ];

        return $this->successResponse($data, __('messages.about_us_retrieved'));
    }

    /**
     * تحديث بيانات من نحن
     */
    public function updateAboutUs(Request $request)
    {
        try {
            $validated = $request->validate([
                'about_us_ar' => 'nullable|string',
                'about_us_en' => 'nullable|string',
                'message_ar' => 'nullable|string',
                'message_en' => 'nullable|string',
                'vision_ar' => 'nullable|string',
                'vision_en' => 'nullable|string',
                'values' => 'nullable|array',
                'values.*.title_ar' => 'required_with:values.*|string|max:255',
                'values.*.title_en' => 'required_with:values.*|string|max:255',
                'values.*.description_ar' => 'nullable|string',
                'values.*.description_en' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        $updateData = [];

        // تحديث النصوص
        if ($request->has('about_us_ar')) {
            $updateData['about_us_ar'] = $request->about_us_ar;
        }
        if ($request->has('about_us_en')) {
            $updateData['about_us_en'] = $request->about_us_en;
        }
        if ($request->has('message_ar')) {
            $updateData['message_ar'] = $request->message_ar;
        }
        if ($request->has('message_en')) {
            $updateData['message_en'] = $request->message_en;
        }
        if ($request->has('vision_ar')) {
            $updateData['vision_ar'] = $request->vision_ar;
        }
        if ($request->has('vision_en')) {
            $updateData['vision_en'] = $request->vision_en;
        }

        // تحديث القيم (فصلها إلى مصفوفتين)
        if ($request->has('values')) {
            $valuesAr = [];
            $valuesEn = [];

            foreach ($request->values as $value) {
                $valuesAr[] = [
                    'title' => $value['title_ar'],
                    'description' => $value['description_ar'] ?? '',
                ];
                $valuesEn[] = [
                    'title' => $value['title_en'],
                    'description' => $value['description_en'] ?? '',
                ];
            }

            $updateData['values_ar'] = $valuesAr;
            $updateData['values_en'] = $valuesEn;
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'about_us_ar' => $updateData['about_us_ar'] ?? '',
                'about_us_en' => $updateData['about_us_en'] ?? '',
                'message_ar' => $updateData['message_ar'] ?? '',
                'message_en' => $updateData['message_en'] ?? '',
                'vision_ar' => $updateData['vision_ar'] ?? '',
                'vision_en' => $updateData['vision_en'] ?? '',
                'values_ar' => $updateData['values_ar'] ?? [],
                'values_en' => $updateData['values_en'] ?? [],
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update($updateData);
            $generalInfo->refresh();
        }

        // إرجاع البيانات المحدثة
        $valuesEn = $generalInfo->values_en ?? [];
        $valuesAr = $generalInfo->values_ar ?? [];
        
        $values = [];
        $maxLength = max(count($valuesEn), count($valuesAr));
        
        for ($i = 0; $i < $maxLength; $i++) {
            $values[] = [
                'title_ar' => $valuesAr[$i]['title'] ?? '',
                'title_en' => $valuesEn[$i]['title'] ?? '',
                'description_ar' => $valuesAr[$i]['description'] ?? '',
                'description_en' => $valuesEn[$i]['description'] ?? '',
            ];
        }

        $data = [
            'about_us_ar' => $generalInfo->about_us_ar ?? '',
            'about_us_en' => $generalInfo->about_us_en ?? '',
            'message_ar' => $generalInfo->message_ar ?? '',
            'message_en' => $generalInfo->message_en ?? '',
            'vision_ar' => $generalInfo->vision_ar ?? '',
            'vision_en' => $generalInfo->vision_en ?? '',
            'values' => $values,
        ];

        return $this->successResponse($data, __('messages.about_us_updated'));
    }

    /**
     * عرض الشروط والأحكام
     */
    public function getTermsAndConditions()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'terms_and_conditions_en' => '',
                'terms_and_conditions_ar' => '',
            ]);
        }

        $data = [
            'terms_and_conditions_ar' => $generalInfo->terms_and_conditions_ar ?? '',
            'terms_and_conditions_en' => $generalInfo->terms_and_conditions_en ?? '',
        ];

        return $this->successResponse($data, __('messages.terms_and_conditions_retrieved'));
    }

    /**
     * تحديث الشروط والأحكام
     */
    public function updateTermsAndConditions(Request $request)
    {
        try {
            $validated = $request->validate([
                'terms_and_conditions_ar' => 'nullable|string',
                'terms_and_conditions_en' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        $updateData = [];

        if ($request->has('terms_and_conditions_ar')) {
            $updateData['terms_and_conditions_ar'] = $request->terms_and_conditions_ar;
        }
        if ($request->has('terms_and_conditions_en')) {
            $updateData['terms_and_conditions_en'] = $request->terms_and_conditions_en;
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'terms_and_conditions_ar' => $updateData['terms_and_conditions_ar'] ?? '',
                'terms_and_conditions_en' => $updateData['terms_and_conditions_en'] ?? '',
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update($updateData);
            $generalInfo->refresh();
        }

        $data = [
            'terms_and_conditions_ar' => $generalInfo->terms_and_conditions_ar ?? '',
            'terms_and_conditions_en' => $generalInfo->terms_and_conditions_en ?? '',
        ];

        return $this->successResponse($data, __('messages.terms_and_conditions_updated'));
    }

    /**
     * عرض سياسة الخصوصية
     */
    public function getPrivacyPolicy()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'privacy_policy_en' => '',
                'privacy_policy_ar' => '',
            ]);
        }

        $data = [
            'privacy_policy_ar' => $generalInfo->privacy_policy_ar ?? '',
            'privacy_policy_en' => $generalInfo->privacy_policy_en ?? '',
        ];

        return $this->successResponse($data, __('messages.privacy_policy_retrieved'));
    }

    /**
     * تحديث سياسة الخصوصية
     */
    public function updatePrivacyPolicy(Request $request)
    {
        try {
            $validated = $request->validate([
                'privacy_policy_ar' => 'nullable|string',
                'privacy_policy_en' => 'nullable|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        $updateData = [];

        if ($request->has('privacy_policy_ar')) {
            $updateData['privacy_policy_ar'] = $request->privacy_policy_ar;
        }
        if ($request->has('privacy_policy_en')) {
            $updateData['privacy_policy_en'] = $request->privacy_policy_en;
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                'privacy_policy_ar' => $updateData['privacy_policy_ar'] ?? '',
                'privacy_policy_en' => $updateData['privacy_policy_en'] ?? '',
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update($updateData);
            $generalInfo->refresh();
        }

        $data = [
            'privacy_policy_ar' => $generalInfo->privacy_policy_ar ?? '',
            'privacy_policy_en' => $generalInfo->privacy_policy_en ?? '',
        ];

        return $this->successResponse($data, __('messages.privacy_policy_updated'));
    }

    /**
     * عرض الإحصائيات
     */
    public function getStatistics()
    {
        $generalInfo = GeneralInfo::first();

        if (!$generalInfo) {
            // إنشاء سجل افتراضي إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                 'contracts_counter' => '0',
                'support_time' => '0',
                'experience_years' => '0',
                'happy_clients' => '0',
                'success_partners' => '0',
            ]);
        }

        $data = [
             'contracts_counter' => $generalInfo->contracts_counter ?? '0',
            'support_time' => $generalInfo->support_time ?? '0',
            'experience_years' => $generalInfo->experience_years ?? '0',
            'happy_clients' => $generalInfo->happy_clients ?? '0',
            'success_partners' => $generalInfo->success_partners ?? '0',
        ];

        return $this->successResponse($data, __('messages.statistics_retrieved'));
    }

    /**
     * تحديث الإحصائيات
     */
    public function updateStatistics(Request $request)
    {
        try {
            $validated = $request->validate([
                 'contracts_counter' => 'nullable|string|max:255',
                'support_time' => 'nullable|string|max:255',
                'experience_years' => 'nullable|string|max:255',
                'happy_clients' => 'nullable|string|max:255',
                'success_partners' => 'nullable|string|max:255',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $generalInfo = GeneralInfo::first();

        $updateData = [];
 
        if ($request->has('contracts_counter')) {
            $updateData['contracts_counter'] = $request->contracts_counter;
        }
        if ($request->has('support_time')) {
            $updateData['support_time'] = $request->support_time;
        }
        if ($request->has('experience_years')) {
            $updateData['experience_years'] = $request->experience_years;
        }
        if ($request->has('happy_clients')) {
            $updateData['happy_clients'] = $request->happy_clients;
        }
        if ($request->has('success_partners')) {
            $updateData['success_partners'] = $request->success_partners;
        }

        if (!$generalInfo) {
            // إنشاء سجل جديد إذا لم يكن موجوداً
            $generalInfo = GeneralInfo::create([
                 'contracts_counter' => $updateData['contracts_counter'] ?? '0',
                'support_time' => $updateData['support_time'] ?? '0',
                'experience_years' => $updateData['experience_years'] ?? '0',
                'happy_clients' => $updateData['happy_clients'] ?? '0',
                'success_partners' => $updateData['success_partners'] ?? '0',
            ]);
        } else {
            // تحديث البيانات الموجودة
            $generalInfo->update($updateData);
            $generalInfo->refresh();
        }

        $data = [
             'contracts_counter' => $generalInfo->contracts_counter ?? '0',
            'support_time' => $generalInfo->support_time ?? '0',
            'experience_years' => $generalInfo->experience_years ?? '0',
            'happy_clients' => $generalInfo->happy_clients ?? '0',
            'success_partners' => $generalInfo->success_partners ?? '0',
        ];

        return $this->successResponse($data, __('messages.statistics_updated'));
    }
}
