<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\FAQ;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class FAQController extends Controller
{
    use ApiResponse;

    /**
     * عرض قائمة الأسئلة الشائعة
     */
    public function index(Request $request)
    {
        $query = FAQ::query();

        // البحث
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('question_ar', 'like', "%{$search}%")
                  ->orWhere('question_en', 'like', "%{$search}%")
                  ->orWhere('answer_ar', 'like', "%{$search}%")
                  ->orWhere('answer_en', 'like', "%{$search}%");
            });
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Pagination
        $perPage = $request->get('per_page', 15);
        $faqs = $query->paginate($perPage);

        return $this->successResponse($faqs, __('messages.faqs_listed'));
    }

    /**
     * إنشاء سؤال جديد
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'question_ar' => 'required|string|max:255',
                'question_en' => 'required|string|max:255',
                'answer_ar' => 'required|string',
                'answer_en' => 'required|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $faq = FAQ::create([
            'question_ar' => $request->question_ar,
            'question_en' => $request->question_en,
            'answer_ar' => $request->answer_ar,
            'answer_en' => $request->answer_en,
        ]);

        return $this->successResponse([
            'faq' => $faq,
        ], __('messages.faq_created'));
    }

    /**
     * عرض سؤال محدد
     */
    public function show($id)
    {
        $faq = FAQ::find($id);

        if (!$faq) {
            return $this->notFoundResponse(__('messages.faq_not_found'));
        }

        return $this->successResponse($faq, __('messages.faq_retrieved'));
    }

    /**
     * تحديث سؤال
     */
    public function update(Request $request, $id)
    {
        $faq = FAQ::find($id);

        if (!$faq) {
            return $this->notFoundResponse(__('messages.faq_not_found'));
        }

        try {
            $validated = $request->validate([
                'question_ar' => 'sometimes|required|string|max:255',
                'question_en' => 'sometimes|required|string|max:255',
                'answer_ar' => 'sometimes|required|string',
                'answer_en' => 'sometimes|required|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        $faq->update($request->only(['question_ar', 'question_en', 'answer_ar', 'answer_en']));

        return $this->successResponse([
            'faq' => $faq,
        ], __('messages.faq_updated'));
    }

    /**
     * حذف سؤال
     */
    public function destroy($id)
    {
        $faq = FAQ::find($id);

        if (!$faq) {
            return $this->notFoundResponse(__('messages.faq_not_found'));
        }

        $faq->delete();

        return $this->successResponse(null, __('messages.faq_deleted'));
    }
}
