<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class CityController extends Controller
{
    use ApiResponse;

    /**
     * عرض قائمة المدن
     */
    public function index(Request $request)
    {
        $query = City::withCount(['zones', 'users']);

        // البحث
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name_ar', 'like', "%{$search}%")
                  ->orWhere('name_en', 'like', "%{$search}%");
            });
        }

        // الترتيب
        $sortBy = $request->get('sort_by', 'created_at');
        $sortOrder = $request->get('sort_order', 'desc');
        
        // السماح بالترتيب حسب name (سيستخدم accessor)
        if ($sortBy === 'name') {
            $query->orderBy('name_ar', $sortOrder);
        } else {
            $query->orderBy($sortBy, $sortOrder);
        }

        // Pagination
        $perPage = $request->get('per_page', 15);
        $cities = $query->paginate($perPage);

        return $this->successResponse($cities, __('messages.cities_listed'));
    }

    /**
     * إنشاء مدينة جديدة
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // التحقق من عدم وجود مدينة بنفس الاسم
        $existingCity = City::where('name_ar', $request->name_ar)
            ->orWhere('name_en', $request->name_en)
            ->first();

        if ($existingCity) {
            return $this->errorResponse(__('messages.city_already_exists'), 409);
        }

        $city = City::create([
            'name_ar' => $request->name_ar,
            'name_en' => $request->name_en,
        ]);

        return $this->successResponse([
            'city' => $city,
        ], __('messages.city_created'));
    }

    /**
     * عرض مدينة محددة
     */
    public function show($id)
    {
        $city = City::with(['zones', 'users'])->withCount(['zones', 'users'])->find($id);

        if (!$city) {
            return $this->notFoundResponse(__('messages.city_not_found'));
        }

        return $this->successResponse($city, __('messages.city_retrieved'));
    }

    /**
     * تحديث مدينة
     */
    public function update(Request $request, $id)
    {
        $city = City::find($id);

        if (!$city) {
            return $this->notFoundResponse(__('messages.city_not_found'));
        }

        try {
            $validated = $request->validate([
                'name_ar' => 'sometimes|required|string|max:255',
                'name_en' => 'sometimes|required|string|max:255',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // التحقق من عدم وجود مدينة أخرى بنفس الاسم
        if ($request->has('name_ar') || $request->has('name_en')) {
            $existingCity = City::where(function($q) use ($request, $id) {
                if ($request->has('name_ar')) {
                    $q->where('name_ar', $request->name_ar);
                }
                if ($request->has('name_en')) {
                    $q->orWhere('name_en', $request->name_en);
                }
            })
            ->where('id', '!=', $id)
            ->first();

            if ($existingCity) {
                return $this->errorResponse(__('messages.city_already_exists'), 409);
            }
        }

        $city->update($request->only(['name_ar', 'name_en']));

        return $this->successResponse([
            'city' => $city->load(['zones', 'users'])->loadCount(['zones', 'users']),
        ], __('messages.city_updated'));
    }

    /**
     * حذف مدينة
     */
    public function destroy($id)
    {
        $city = City::withCount(['zones', 'users'])->find($id);

        if (!$city) {
            return $this->notFoundResponse(__('messages.city_not_found'));
        }

        // التحقق من وجود مناطق مرتبطة
        if ($city->zones_count > 0) {
            return $this->errorResponse(__('messages.city_has_zones'), 400);
        }

        // التحقق من وجود مستخدمين مرتبطين
        if ($city->users_count > 0) {
            return $this->errorResponse(__('messages.city_has_users'), 400);
        }

        $city->delete();

        return $this->successResponse(null, __('messages.city_deleted'));
    }
}
