<?php

namespace App\Http\Controllers\API\Dashboard;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Mail\ResetPasswordMail;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    use ApiResponse;

    /**
     * تسجيل دخول الأدمن
     */
    public function login(Request $request)
    {
        // التحقق من البيانات المدخلة
        try {
            $validated = $request->validate([
                'email' => 'required|email',
                'password' => 'required|string',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // البحث عن المستخدم بالبريد الإلكتروني
        $user = User::where('email', $request->email)->first();

        // التحقق من وجود المستخدم وكلمة المرور
        if (!$user || !Hash::check($request->password, $user->password)) {
            return $this->errorResponse(
                __('messages.login_failed'),
                401
            );
        }

        // التحقق من نوع المستخدم (يجب أن يكون 'user')
        if ($user->type !== 'user') {
            return $this->errorResponse(
                __('messages.login_not_admin'),
                403
            );
        }

        // التحقق من حالة المستخدم (يجب أن يكون 'active')
        if ($user->status !== 'active') {
            return $this->errorResponse(
                __('messages.login_inactive'),
                403
            );
        }

        // تحديث آخر وقت تسجيل دخول
        $user->update([
            'last_login_at' => now(),
        ]);

        // إنشاء token للمستخدم
        $token = $user->createToken('admin-token')->plainTextToken;

        // جلب الأدوار والصلاحيات للمستخدم
        $roles = $user->getRoleNames();
        $permissions = $user->getAllPermissions()->pluck('name');

        // إرجاع البيانات مع الـ token
        return $this->successResponse([
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'type' => $user->type,
                'status' => $user->status,
                'roles' => $roles,
                'permissions' => $permissions,
            ],
            'token' => $token,
        ], __('messages.login_success'));
    }

    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();
        $request->user()->tokens()->delete();
        return $this->successResponse(null, __('messages.logout_success'));
    }
    public function changePassword(Request $request)
    {
        $validated = $request->validate([
            'old_password' => 'required|string',
            'new_password' => 'required|string|confirmed',
        ]);
        $user = $request->user();
        if (!Hash::check($request->old_password, $user->password)) {
            return $this->errorResponse(
                __('messages.old_password_incorrect'),
                401
            );
        }
        $user->password = Hash::make($request->new_password);
        $user->save();
        return $this->successResponse(null, __('messages.password_changed_success'));
    }

    /**
     * طلب إعادة تعيين كلمة المرور
     */
    public function forgotPassword(Request $request)
    {
        try {
            $validated = $request->validate([
                'email' => 'required|email',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // البحث عن المستخدم بالبريد الإلكتروني
        $user = User::where('email', $request->email)->first();

        // إذا لم يتم العثور على المستخدم، نرجع نفس الرسالة لأسباب أمنية
        if (!$user) {
            return $this->errorResponse(
                __('messages.forgot_password_email_not_found'),
                404
            );
        }

        // التحقق من نوع المستخدم (يجب أن يكون 'user')
        if ($user->type !== 'user') {
            return $this->errorResponse(
                __('messages.login_not_admin'),
                403
            );
        }

        // إنشاء كود OTP عشوائي (6 أرقام)
        $otpCode = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);

        // تحديث كود OTP وتاريخ انتهاء الصلاحية (15 دقيقة)
        $user->update([
            'otp_code' => $otpCode,
            'otp_code_expires_at' => now()->addMinutes(15),
        ]);

        // إرسال الإيميل مع كود OTP
        try {
            Mail::to($user->email)->send(new ResetPasswordMail($otpCode, $user->name));
        } catch (\Exception $e) {
            // في حالة فشل الإرسال، نسجل الخطأ لكن نستمر في العملية
            // في التطوير، قد لا تكون إعدادات البريد الإلكتروني موجودة
            \Log::error('Failed to send reset password email: ' . $e->getMessage());
            \Log::error('Exception details: ' . $e->getTraceAsString());
            
            // في بيئة التطوير، نرجع OTP في الاستجابة للاختبار
            if (config('app.debug') || config('app.env') === 'local') {
                return $this->successResponse(
                    ['otp_code' => $otpCode], // فقط في التطوير للاختبار
                    __('messages.forgot_password_success') . ' (OTP: ' . $otpCode . ')'
                );
            }
            
            // في الإنتاج، نرجع رسالة عامة بدون كشف OTP
            return $this->errorResponse(
                __('messages.error'),
                500
            );
        }

        return $this->successResponse(
            null,
            __('messages.forgot_password_success')
        );
    }

    /**
     * إعادة تعيين كلمة المرور باستخدام كود OTP
     */
    public function resetPassword(Request $request)
    {
        try {
            $validated = $request->validate([
                'email' => 'required|email',
                'reset_code' => 'required|string|size:6',
                'new_password' => 'required|string|min:8|confirmed',
            ]);
        } catch (ValidationException $e) {
            return $this->validationErrorResponse($e->errors());
        }

        // البحث عن المستخدم بالبريد الإلكتروني
        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return $this->errorResponse(
                __('messages.forgot_password_email_not_found'),
                404
            );
        }

        // التحقق من كود OTP
        if (!$user->otp_code || $user->otp_code !== $request->reset_code) {
            return $this->errorResponse(
                __('messages.reset_code_invalid'),
                400
            );
        }

        // التحقق من انتهاء صلاحية كود OTP
        if (!$user->otp_code_expires_at || now()->greaterThan($user->otp_code_expires_at)) {
            return $this->errorResponse(
                __('messages.reset_code_invalid'),
                400
            );
        }

        // تحديث كلمة المرور
        $user->update([
            'password' => Hash::make($request->new_password),
            'otp_code' => null,
            'otp_code_expires_at' => null,
            'otp_code_verified_at' => now()->toDateTimeString(),
        ]);

        return $this->successResponse(
            null,
            __('messages.reset_password_success')
        );
    }
}
