<?php

namespace App\Http\Controllers\API\Client;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use App\Traits\TwilioTrait;
use App\Models\User;
use App\Models\PasswordResetOtp;
use App\Models\ProfileUpdateRequest;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Models\Role;

class AuthController extends Controller
{
    use ApiResponse, TwilioTrait;

    /**
     * إنشاء حساب أولي (رقم التلفون وكلمة المرور فقط)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function registerInitial(Request $request)
    {
        try {
            // التحقق من البيانات
            $validator = Validator::make($request->all(), [
                'mobile' => 'required|string|regex:/^\+[1-9]\d{1,14}$/|unique:users,phone',
                'password' => 'required|string|min:8',
            ], [
                'mobile.required' => 'رقم الهاتف مطلوب',
                'mobile.regex' => 'رقم الهاتف يجب أن يبدأ برمز الدولة (مثال: +966)',
                'mobile.unique' => 'رقم الهاتف موجود بالفعل',
                'password.required' => 'كلمة المرور مطلوبة',
                'password.min' => 'كلمة المرور يجب أن تكون 8 أحرف على الأقل',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            // التحقق من أن رقم الهاتف غير مكرر
            $existingUser = User::where('phone', $request->mobile)->first();
            if ($existingUser) {
                return $this->errorResponse('رقم الهاتف موجود بالفعل في النظام', 422);
            }

            // إنشاء كود OTP عشوائي (6 أرقام)
            $otpCode = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);

            // إنشاء المستخدم (بيانات أولية فقط)
            $user = User::create([
                'name' => 'Pending',
                'middle_name' => '',
                'third_name' => '',
                'last_name' => '',
                'phone' => $request->mobile,
                'whatsapp' => $request->mobile,
                'password' => Hash::make($request->password),
                'status' => 'inactive', // غير نشط حتى يتم التحقق من OTP واستكمال البيانات
                'is_verified' => false,
                'otp_code' => $otpCode,
                'otp_code_expires_at' => now()->addMinutes(15), // كود OTP صالح لمدة 15 دقيقة
            ]);

            // إرسال كود التحقق عبر SMS باستخدام Twilio
            try {
                $this->SendSms($user, $otpCode);
            } catch (\Exception $e) {
                \Log::error('Failed to send SMS via Twilio: ' . $e->getMessage());
                if (config('app.debug') || config('app.env') === 'local') {
                    \Log::warning('OTP Code (for testing only): ' . $otpCode);
                }
            }

            // إرجاع البيانات الأساسية
            $userData = [
                'id' => $user->id,
                'phone' => $user->phone,
            ];

            return $this->successResponse($userData, 'تم إنشاء الحساب بنجاح. يرجى التحقق من رقم الهاتف باستخدام الكود المرسل إليك', 201);

        } catch (\Illuminate\Database\QueryException $e) {
            return $this->errorResponse('حدث خطأ في قاعدة البيانات: ' . $e->getMessage(), 500);
        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء إنشاء الحساب: ' . $e->getMessage(), 500);
        }
    }

   

    /**
     * التحقق من كود OTP
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function verifyOTP(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'otp_code' => 'required|string|size:6',
            ], [
                'user_id.required' => 'معرف المستخدم مطلوب',
                'user_id.exists' => 'المستخدم غير موجود',
                'otp_code.required' => 'كود التحقق مطلوب',
                'otp_code.size' => 'كود التحقق يجب أن يكون 6 أرقام',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            $user = User::findOrFail($request->user_id);

            // التحقق من كود OTP
            if (!$user->otp_code || $user->otp_code !== $request->otp_code) {
                return $this->errorResponse('كود التحقق غير صحيح', 400);
            }

            // التحقق من انتهاء صلاحية كود OTP
            if (!$user->otp_code_expires_at || now()->greaterThan($user->otp_code_expires_at)) {
                return $this->errorResponse('كود التحقق منتهي الصلاحية', 400);
            }

            // تحديث حالة التحقق
            $user->update([
                'otp_code_verified_at' => now(),
            ]);

            return $this->successResponse([
                'user_id' => $user->id,
                'phone' => $user->phone,
            ], 'تم التحقق من الرقم بنجاح', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء التحقق من الكود: ' . $e->getMessage(), 500);
        }
    }

    /**
     * استكمال بيانات الملف الشخصي
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function completeProfile(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'firstName' => 'required|string|max:255',
                'secondName' => 'required|string|max:255',
                'thirdName' => 'required|string|max:255',
                'lastName' => 'required|string|max:255',
                'idType' => 'required|in:national,resident,passport',
                'idNumber' => 'required|string|regex:/^[0-9]+$/',
                'email' => 'nullable|email|max:255|unique:users,email',
                'city_id' => 'nullable|exists:cities,id',
                'zone_id' => 'nullable|exists:zones,id',
                'street' => 'nullable|string|max:255',
                'role' => 'required|in:landlord,tenant',
                'nationality' => 'nullable|string|max:255',
            ], [
                'user_id.required' => 'معرف المستخدم مطلوب',
                'user_id.exists' => 'المستخدم غير موجود',
                'firstName.required' => 'الاسم الأول مطلوب',
                'secondName.required' => 'الاسم الثاني مطلوب',
                'thirdName.required' => 'الاسم الثالث مطلوب',
                'lastName.required' => 'اللقب مطلوب',
                'idType.required' => 'نوع الهوية مطلوب',
                'idType.in' => 'نوع الهوية يجب أن يكون: national, resident, passport',
                'idNumber.required' => 'رقم الهوية مطلوب',
                'idNumber.regex' => 'رقم الهوية يجب أن يحتوي على أرقام فقط',
                'email.email' => 'البريد الإلكتروني غير صحيح',
                'email.unique' => 'البريد الإلكتروني موجود بالفعل',
                'role.required' => 'نوع المستخدم مطلوب',
                'role.in' => 'نوع المستخدم يجب أن يكون: landlord (مؤجر) أو tenant (مستأجر)',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            $user = User::findOrFail($request->user_id);

            // التحقق من أن المستخدم قد تحقق من OTP
            if (!$user->otp_code_verified_at) {
                return $this->errorResponse('يجب التحقق من رقم الهاتف أولاً', 400);
            }

            // تحديث بيانات المستخدم
            $user->update([
                'name' => trim($request->firstName),
                'middle_name' => trim($request->secondName),
                'third_name' => trim($request->thirdName),
                'last_name' => trim($request->lastName),
                'id_type' => $request->idType,
                'national_id' => $request->idNumber,
                'email' => $request->email ? trim($request->email) : null,
                'nationality' => $request->nationality ? trim($request->nationality) : null,
                'city_id' => $request->city_id ? (int)$request->city_id : null,
                'zone_id' => $request->zone_id ? (int)$request->zone_id : null,
                'address' => $request->street ? trim($request->street) : null,
                'type' => $request->role,
                'status' => 'active',
                'is_verified' => true,
            ]);

            // تحميل العلاقات
            $user->load(['city', 'zone']);

            // إرجاع البيانات الكاملة
            $userData = [
                'id' => $user->id,
                'name' => $user->name ?? '',
                'middle_name' => $user->middle_name ?? '',
                'third_name' => $user->third_name ?? '',
                'last_name' => $user->last_name ?? '',
                'full_name' => trim(($user->name ?? '') . ' ' . ($user->middle_name ?? '') . ' ' . ($user->third_name ?? '') . ' ' . ($user->last_name ?? '')),
                'phone' => $user->phone ?? '',
                'email' => $user->email ?? '',
                'whatsapp' => $user->whatsapp ?? '',
                'national_id' => $user->national_id ?? '',
                'id_type' => $user->id_type ?? '',
                'nationality' => $user->nationality ?? '',
                'type' => $user->type ?? '',
                'status' => $user->status ?? '',
                'gender' => $user->gender ?? '',
                'address' => $user->address ?? '',
                'city_id' => $user->city_id,
                'zone_id' => $user->zone_id,
                'is_verified' => $user->is_verified ?? false,
            ];

            return $this->successResponse($userData, 'تم استكمال بيانات الحساب بنجاح', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء استكمال البيانات: ' . $e->getMessage(), 500);
        }
    }

    /**
     * إعادة إرسال كود OTP
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resendOTP(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
            ], [
                'user_id.required' => 'معرف المستخدم مطلوب',
                'user_id.exists' => 'المستخدم غير موجود',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            $user = User::findOrFail($request->user_id);

            // التحقق من السياق: هل هو للتسجيل أم لإعادة تعيين كلمة المرور؟
            // إذا كان هناك password_reset_otp غير verified، فهو لإعادة تعيين كلمة المرور
            $passwordResetOtp = PasswordResetOtp::forUser($user->id)
                ->whereNull('verified_at')
                ->latest()
                ->first();

            if ($passwordResetOtp) {
                // السياق: إعادة تعيين كلمة المرور - استخدام password_reset_otps table
                // حذف OTP القديم
                $passwordResetOtp->delete();

                // إنشاء كود OTP جديد (6 أرقام)
                $otpCode = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);

                // إنشاء OTP جديد
                $newPasswordResetOtp = PasswordResetOtp::create([
                    'user_id' => $user->id,
                    'phone' => $user->phone,
                    'otp_code' => $otpCode,
                    'expires_at' => now()->addMinutes(15),
                ]);

                // إرسال كود التحقق عبر SMS باستخدام Twilio
                try {
                    $this->SendSms($user, $otpCode);
                } catch (\Exception $e) {
                    \Log::error('Failed to send SMS via Twilio: ' . $e->getMessage());
                    // حذف OTP الذي تم إنشاؤه إذا فشل الإرسال
                    $newPasswordResetOtp->delete();
                    
                    if (config('app.debug') || config('app.env') === 'local') {
                        \Log::warning('OTP Code (for testing only): ' . $otpCode);
                    }
                    return $this->errorResponse('حدث خطأ أثناء إرسال كود التحقق', 500);
                }

                return $this->successResponse([
                    'user_id' => $user->id,
                    'phone' => $user->phone,
                ], 'تم إعادة إرسال كود التحقق بنجاح', 200);
            } else {
                // السياق: التسجيل - استخدام users table
                // التحقق من أن المستخدم لم يتم التحقق منه بعد (pending)
                if ($user->otp_code_verified_at || $user->name !== 'Pending') {
                    return $this->errorResponse('تم التحقق من هذا المستخدم بالفعل', 400);
                }

                // إنشاء كود OTP جديد (6 أرقام)
                $otpCode = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);

                // تحديث كود OTP وتاريخ انتهاء الصلاحية (15 دقيقة)
                $user->update([
                    'otp_code' => $otpCode,
                    'otp_code_expires_at' => now()->addMinutes(15),
                    'otp_code_verified_at' => null, // إعادة تعيين حالة التحقق
                ]);

                // إرسال كود التحقق عبر SMS باستخدام Twilio
                try {
                    $this->SendSms($user, $otpCode);
                } catch (\Exception $e) {
                    \Log::error('Failed to send SMS via Twilio: ' . $e->getMessage());
                    if (config('app.debug') || config('app.env') === 'local') {
                        \Log::warning('OTP Code (for testing only): ' . $otpCode);
                    }
                }

                return $this->successResponse([
                    'user_id' => $user->id,
                    'phone' => $user->phone,
                ], 'تم إعادة إرسال كود التحقق بنجاح', 200);
            }

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء إعادة إرسال الكود: ' . $e->getMessage(), 500);
        }
    }

    /**
     * تسجيل دخول العميل
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        try {
            // التحقق من البيانات
            $validator = Validator::make($request->all(), [
                'mobile' => 'required|string|regex:/^\+[1-9]\d{1,14}$/',
                'password' => 'required|string',
            ], [
                'mobile.required' => 'رقم الهاتف مطلوب',
                'mobile.regex' => 'رقم الهاتف يجب أن يبدأ برمز الدولة (مثال: +966)',
                'password.required' => 'كلمة المرور مطلوبة',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            // البحث عن المستخدم برقم الهاتف
            $user = User::where('phone', $request->mobile)->first();

            // التحقق من وجود المستخدم وكلمة المرور
            if (!$user || !Hash::check($request->password, $user->password)) {
                return $this->errorResponse('رقم الهاتف أو كلمة المرور غير صحيحة', 401);
            }

            // التحقق من نوع المستخدم (يجب أن يكون 'landlord' أو 'tenant')
            if (!in_array($user->type, ['landlord', 'tenant'])) {
                return $this->errorResponse('هذا الحساب غير مصرح له بالدخول', 403);
            }

            // التحقق من حالة المستخدم (يجب أن يكون 'active')
            if ($user->status !== 'active') {
                return $this->errorResponse('الحساب غير نشط. يرجى التواصل مع الدعم الفني', 403);
            }

            // التحقق من أن المستخدم قد أكمل بياناته (ليس pending)
            if ($user->name === 'Pending' || !$user->is_verified) {
                return $this->errorResponse('يرجى إكمال بيانات الحساب أولاً', 403);
            }

            // تحديث آخر وقت تسجيل دخول
            $user->update([
                'last_login_at' => now(),
            ]);

            // إنشاء token للمستخدم
            $token = $user->createToken('client-token')->plainTextToken;

            // تحميل العلاقات
            $user->load(['city', 'zone']);

            // إرجاع البيانات مع الـ token
            $userData = [
                'id' => $user->id,
                'name' => $user->name ?? '',
                'middle_name' => $user->middle_name ?? '',
                'third_name' => $user->third_name ?? '',
                'last_name' => $user->last_name ?? '',
                'full_name' => trim(($user->name ?? '') . ' ' . ($user->middle_name ?? '') . ' ' . ($user->third_name ?? '') . ' ' . ($user->last_name ?? '')),
                'phone' => $user->phone ?? '',
                'email' => $user->email ?? '',
                'whatsapp' => $user->whatsapp ?? '',
                'national_id' => $user->national_id ?? '',
                'id_type' => $user->id_type ?? '',
                'nationality' => $user->nationality ?? '',
                'type' => $user->type ?? '',
                'status' => $user->status ?? '',
                'gender' => $user->gender ?? '',
                'address' => $user->address ?? '',
                'city_id' => $user->city_id,
                'zone_id' => $user->zone_id,
                'is_verified' => $user->is_verified ?? false,
            ];

            return $this->successResponse([
                'user' => $userData,
                'token' => $token,
            ], 'تم تسجيل الدخول بنجاح', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء تسجيل الدخول: ' . $e->getMessage(), 500);
        }
    }

    /**
     * طلب إعادة تعيين كلمة المرور
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function forgotPassword(Request $request)
    {
        try {
            // التحقق من البيانات
            $validator = Validator::make($request->all(), [
                'mobile' => 'required|string|regex:/^\+[1-9]\d{1,14}$/',
            ], [
                'mobile.required' => 'رقم الهاتف مطلوب',
                'mobile.regex' => 'رقم الهاتف يجب أن يبدأ برمز الدولة (مثال: +966)',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            // البحث عن المستخدم برقم الهاتف
            $user = User::where('phone', $request->mobile)->first();

            // إذا لم يتم العثور على المستخدم، نرجع نفس الرسالة لأسباب أمنية
            if (!$user) {
                return $this->errorResponse('لا يوجد حساب مرتبط برقم الهاتف هذا', 404);
            }

            // التحقق من نوع المستخدم (يجب أن يكون 'landlord' أو 'tenant')
            if (!in_array($user->type, ['landlord', 'tenant'])) {
                return $this->errorResponse('هذا الحساب غير مصرح له باستخدام هذه الخدمة', 403);
            }

            // إنشاء كود OTP عشوائي (6 أرقام)
            $otpCode = str_pad(rand(0, 999999), 6, '0', STR_PAD_LEFT);

            // إنشاء سجل جديد في جدول password_reset_otps
            // حذف أي OTPs سابقة غير مستخدمة للمستخدم نفسه
            PasswordResetOtp::where('user_id', $user->id)
                ->whereNull('verified_at')
                ->delete();

            // إنشاء OTP جديد
            $passwordResetOtp = PasswordResetOtp::create([
                'user_id' => $user->id,
                'phone' => $user->phone,
                'otp_code' => $otpCode,
                'expires_at' => now()->addMinutes(15),
            ]);

            // إرسال كود التحقق عبر SMS باستخدام Twilio
         /*   try {
                $this->SendSms($user, $otpCode);
            } catch (\Exception $e) {
                \Log::error('Failed to send SMS via Twilio: ' . $e->getMessage());
                // حذف OTP الذي تم إنشاؤه إذا فشل الإرسال
                $passwordResetOtp->delete();
                
                if (config('app.debug') || config('app.env') === 'local') {
                    \Log::warning('OTP Code (for testing only): ' . $otpCode);
                    // في بيئة التطوير، نرجع OTP في الاستجابة للاختبار
                    return $this->successResponse(
                        ['otp_code' => $otpCode, 'user_id' => $user->id],
                        'تم إرسال كود التحقق بنجاح (OTP: ' . $otpCode . ')',
                        200
                    );
                }
                return $this->errorResponse('حدث خطأ أثناء إرسال كود التحقق', 500);
            }*/

            return $this->successResponse([
                'user_id' => $user->id,
                'phone' => $user->phone,
            ], 'تم إرسال كود التحقق إلى رقم الهاتف الخاص بك', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء طلب إعادة تعيين كلمة المرور: ' . $e->getMessage(), 500);
        }
    }

    /**
     * إعادة تعيين كلمة المرور باستخدام كود OTP
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request)
    {
        try {
            // التحقق من البيانات
            $validator = Validator::make($request->all(), [
                'user_id' => 'required|exists:users,id',
                'otp_code' => 'required|string|size:6',
                'new_password' => 'required|string|min:8|confirmed',
            ], [
                'user_id.required' => 'معرف المستخدم مطلوب',
                'user_id.exists' => 'المستخدم غير موجود',
                'otp_code.required' => 'كود التحقق مطلوب',
                'otp_code.size' => 'كود التحقق يجب أن يكون 6 أرقام',
                'new_password.required' => 'كلمة المرور الجديدة مطلوبة',
                'new_password.min' => 'كلمة المرور يجب أن تكون 8 أحرف على الأقل',
                'new_password.confirmed' => 'تأكيد كلمة المرور غير متطابق',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            $user = User::findOrFail($request->user_id);

            // البحث عن OTP صالح في جدول password_reset_otps
            $passwordResetOtp = PasswordResetOtp::forUser($user->id)
                ->where('otp_code', $request->otp_code)
                ->valid()
                ->latest()
                ->first();

            // التحقق من كود OTP
            if (!$passwordResetOtp) {
                return $this->errorResponse('كود التحقق غير صحيح أو منتهي الصلاحية', 400);
            }

            // تحديث كلمة المرور
            $user->update([
                'password' => Hash::make($request->new_password),
            ]);

            // تعليم OTP كـ verified
            $passwordResetOtp->markAsVerified();

            // حذف جميع OTPs الأخرى غير المستخدمة للمستخدم
            PasswordResetOtp::forUser($user->id)
                ->whereNull('verified_at')
                ->where('id', '!=', $passwordResetOtp->id)
                ->delete();

            return $this->successResponse([
                'user_id' => $user->id,
                'phone' => $user->phone,
            ], 'تم إعادة تعيين كلمة المرور بنجاح', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء إعادة تعيين كلمة المرور: ' . $e->getMessage(), 500);
        }
    }

    /**
     * عرض بروفايل المستخدم الحالي
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getProfile(Request $request)
    {
        try {
            $user = $request->user();

            if (!$user) {
                return $this->unauthorizedResponse('غير مصرح لك بالوصول');
            }

            // تحميل العلاقات (city, zone)
            $user->load(['city', 'zone']);

            // جلب بيانات المستخدم مع العلاقات
            $userData = [
                'id' => $user->id,
                'name' => $user->name ?? '',
                'middle_name' => $user->middle_name ?? '',
                'third_name' => $user->third_name ?? '',
                'last_name' => $user->last_name ?? '',
                'full_name' => trim(($user->name ?? '') . ' ' . ($user->middle_name ?? '') . ' ' . ($user->third_name ?? '') . ' ' . ($user->last_name ?? '')),
                'phone' => $user->phone ?? '',
                'email' => $user->email ?? '',
                'whatsapp' => $user->whatsapp ?? '',
                'national_id' => $user->national_id ?? '',
                'id_type' => $user->id_type ?? '',
                'nationality' => $user->nationality ?? '',
                'type' => $user->type ?? '',
                'status' => $user->status ?? '',
                'gender' => $user->gender ?? '',
                'address' => $user->address ?? '',
                'city_id' => $user->city_id,
                'zone_id' => $user->zone_id,
                'city' => $user->city ? [
                    
                'id' => $user->city->id,
                'name' => app()->getLocale() == 'ar' ? ($user->city->name_ar ?? $user->city->name_en ?? ($user->city->name ?? '')) : ($user->city->name_en ?? $user->city->name_ar ?? ($user->city->name ?? '')),
                ] : null,
                'zone' => $user->zone ? [
                    'id' => $user->zone->id,
                    'name' => app()->getLocale() == 'ar' ? ($user->zone->name_ar ?? $user->zone->name_en ?? ($user->zone->name ?? '')) : ($user->zone->name_en ?? $user->zone->name_ar ?? ($user->zone->name ?? '')),
                ] : null,
                'is_verified' => $user->is_verified ?? false,
                'created_at' => $user->created_at,
                'updated_at' => $user->updated_at,
            ];

            return $this->successResponse($userData, 'تم جلب بيانات البروفايل بنجاح', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء جلب بيانات البروفايل: ' . $e->getMessage(), 500);
        }
    }

    /**
     * طلب تحديث بروفايل المستخدم (يحتاج موافقة الأدمن)
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = $request->user();

            if (!$user) {
                return $this->unauthorizedResponse('غير مصرح لك بالوصول');
            }

            // التحقق من البيانات
            $validator = Validator::make($request->all(), [
                'name' => 'sometimes|required|string|max:255',
                'middle_name' => 'sometimes|required|string|max:255',
                'third_name' => 'sometimes|nullable|string|max:255',
                'last_name' => 'sometimes|required|string|max:255',
                'email' => 'sometimes|nullable|email|max:255|unique:users,email,' . $user->id,
                'whatsapp' => 'sometimes|nullable|string|unique:users,whatsapp,' . $user->id,
                'national_id' => 'sometimes|nullable|string|unique:users,national_id,' . $user->id,
                'id_type' => 'sometimes|nullable|in:national,resident,passport',
                'nationality' => 'sometimes|nullable|string|max:255',
                'gender' => 'sometimes|nullable|in:male,female',
                'address' => 'sometimes|nullable|string|max:255',
                'city_id' => 'sometimes|nullable|exists:cities,id',
                'zone_id' => 'sometimes|nullable|exists:zones,id',
            ], [
                'email.unique' => 'البريد الإلكتروني موجود بالفعل',
                'whatsapp.unique' => 'رقم الواتساب موجود بالفعل',
                'national_id.unique' => 'رقم الهوية موجود بالفعل',
                'city_id.exists' => 'المدينة المحددة غير موجودة',
                'zone_id.exists' => 'المنطقة المحددة غير موجودة',
            ]);

            if ($validator->fails()) {
                return $this->validationErrorResponse($validator->errors()->toArray(), 'خطأ في التحقق من البيانات');
            }

            // حفظ البيانات الحالية
            $currentData = [
                'name' => $user->name,
                'middle_name' => $user->middle_name,
                'third_name' => $user->third_name,
                'last_name' => $user->last_name,
                'email' => $user->email,
                'whatsapp' => $user->whatsapp,
                'national_id' => $user->national_id,
                'id_type' => $user->id_type,
                'nationality' => $user->nationality,
                'gender' => $user->gender,
                'address' => $user->address,
                'city_id' => $user->city_id,
                'zone_id' => $user->zone_id,
            ];

            // جمع البيانات المطلوبة للتحديث
            $requestedData = $request->only([
                'name',
                'middle_name',
                'third_name',
                'last_name',
                'email',
                'whatsapp',
                'national_id',
                'id_type',
                'nationality',
                'gender',
                'address',
                'city_id',
                'zone_id',
            ]);

            // إزالة الحقول الفارغة
            $requestedData = array_filter($requestedData, function ($value) {
                return $value !== null && $value !== '';
            });

            // التحقق من وجود تغييرات
            $hasChanges = false;
            foreach ($requestedData as $key => $value) {
                if (isset($currentData[$key]) && $currentData[$key] != $value) {
                    $hasChanges = true;
                    break;
                } elseif (!isset($currentData[$key])) {
                    $hasChanges = true;
                    break;
                }
            }

            if (!$hasChanges) {
                return $this->errorResponse('لم يتم إجراء أي تغييرات على البيانات', 400);
            }

            // التحقق من وجود طلب معلق للمستخدم
            $pendingRequest = ProfileUpdateRequest::where('user_id', $user->id)
                ->where('status', 'pending')
                ->latest()
                ->first();

            if ($pendingRequest) {
                return $this->errorResponse('يوجد طلب تحديث معلق بالفعل. يرجى انتظار المراجعة', 400);
            }

            // إنشاء طلب تحديث البروفايل
            $profileUpdateRequest = ProfileUpdateRequest::create([
                'user_id' => $user->id,
                'requested_data' => $requestedData,
                'current_data' => $currentData,
                'status' => 'pending',
            ]);

            // إرسال إشعار للأدمن
            $this->sendProfileUpdateNotificationToAdmins($user, $profileUpdateRequest);

            return $this->successResponse([
                'request_id' => $profileUpdateRequest->id,
                'status' => $profileUpdateRequest->status,
                'message' => 'تم إرسال طلب تحديث البروفايل بنجاح. سيتم مراجعته من قبل الأدمن',
            ], 'تم إرسال طلب تحديث البروفايل بنجاح. سيتم مراجعته من قبل الأدمن', 200);

        } catch (\Exception $e) {
            return $this->errorResponse('حدث خطأ أثناء طلب تحديث البروفايل: ' . $e->getMessage(), 500);
        }
    }

    /**
     * إرسال إشعار للأدمن عند طلب تحديث البروفايل
     * 
     * @param User $user
     * @param ProfileUpdateRequest $profileUpdateRequest
     * @return void
     */
    private function sendProfileUpdateNotificationToAdmins(User $user, ProfileUpdateRequest $profileUpdateRequest)
    {
        try {
            // الحصول على جميع الأدمن
            $adminRole = Role::where('name', 'الأدمن')->where('guard_name', 'api')->first();
            
            if (!$adminRole) {
                \Log::warning('Admin role not found');
                return;
            }

            $admins = $adminRole->users;

            if ($admins->isEmpty()) {
                \Log::warning('No admin users found');
                return;
            }

            // إرسال إشعار لكل أدمن
            foreach ($admins as $admin) {
                $locale = app()->getLocale();
                
                Notification::create([
                    'user_id' => $admin->id,
                    'title_ar' => 'طلب تحديث بروفايل مستخدم',
                    'title_en' => 'User Profile Update Request',
                    'description_ar' => "المستخدم {$user->name} ({$user->phone}) طلب تحديث بيانات بروفايله. رقم الطلب: #{$profileUpdateRequest->id}",
                    'description_en' => "User {$user->name} ({$user->phone}) requested to update their profile data. Request ID: #{$profileUpdateRequest->id}",
                    'type' => 'profile_update_request',
                    'link' => "/dashboard/profile-update-requests/{$profileUpdateRequest->id}",
                    'is_read' => false,
                    'is_deleted' => false,
                ]);
            }

        } catch (\Exception $e) {
            \Log::error('Failed to send profile update notification to admins: ' . $e->getMessage());
        }
    }
}
